#$Id: traitedDockingParameters.py 258 2015-11-14 17:04:46Z sarkiss $
from enthought.traits.api import HasTraits, List, Trait, Int, Bool, Float, Enum, File, Str, TraitHandler
from enthought.traits.ui.api import Item, Group, View, CheckListEditor, VGroup, HGroup, spring, EnumEditor
from enthought.traits.ui.menu import LiveButtons, UndoButton, RevertButton, OKButton, CancelButton, HelpButton
from AutoDockTools.DockingParameters import DockingParameters
from miscTraits import PositiveInt, TraitPositiveInteger
Buttons = [ UndoButton, RevertButton, OKButton, CancelButton ]
class TraitedDockingParameters( HasTraits ): 
    
    dpo = DockingParameters()
    randomLibraryTuple = ('Platform-Independent library', 'Built-In library')
    randomLibrary = Trait( randomLibraryTuple[0], randomLibraryTuple[1]) 
    
    seedType1 = Trait('time', 'PID', 'custom')
    seedType1.default_value(0,"PID")
    seedType2 = Trait('time', 'PID', 'custom')
    
    seedInt1 = PositiveInt()
    seedInt2 = PositiveInt()


    external_grid_energy = Float(dpo['extnrg']['value'])
    e0max_0 = Float(dpo['e0max']['value'][0])
    e0max_1 = Trait(dpo['e0max']['value'][1], TraitPositiveInteger(), Int)
    calculate_internal_electrostatic_energy = Bool(dpo['intelec']['value'])

    dstep = Float(dpo['dstep']['value'])
    try:
        tstep = Float(dpo['tstep']['value'][0]) #TODO: remove this after PyRx 1.0 release
    except:
        tstep = Float(dpo['tstep']['value'])
    qstep = Float(dpo['qstep']['value'])
    
    
    outputLevelList = ["No output", "Minimal output",
                        "Full state output at end of each cycle",
                        "Detailed output for each step"]
    # The view includes one group per column formation.  These will be displayed
    # on separate tabbed panels.
    
    outputLevelEnum = Enum(outputLevelList, editor = EnumEditor(values = outputLevelList, cols=1))
    outputLevelEnum.default_value = "Minimal output"
    
    write_all_flag = Bool(dpo['write_all_flag']['value'])
    analysis_flag = Bool(dpo['analysis']['value'])    
    rmstol = Float(dpo['rmstol']['value'])
    rmsatoms = Enum("ligand only",'all')
    
    rmsref = File

    custom_parameter_file = Bool(dpo['custom_parameter_file']['value'])
    parameter_file = File(dpo['parameter_file']['value'])

    
    include_1_4_interactions_flag = Bool(dpo['include_1_4_interactions_flag']['value'])
    include_1_4_interactions = Float(dpo['include_1_4_interactions']['value'])
    
    unbound_model_flag = Bool(True)
    unbound_model = Trait("bound", "extended")

    epdb_flag =  Bool(dpo['epdb_flag']['value'])
    epdb = File(dpo['epdb']['value'])

    
    def __init__(self):
        seedType1_group = HGroup(Item(name='seedType1', style='custom', show_label=False), 
                                Group(Item(name='seedInt1', show_label=False), 
                                      enabled_when='object.seedType1 == "custom"'),
                            )    
        
        
        seedType2_group = HGroup(Item(name='seedType2', style='custom', show_label=False), 
                                Group(Item(name='seedInt1', show_label=False), 
                                      enabled_when='object.seedType2 == "custom"'),
                            visible_when = 'randomLibrary == "Platform-Independent library"',
                            )    
        
        seedsGroup = Group( seedType1_group, seedType2_group, 
                           show_border = True,
                           label = 'Random Number Generator Seeds'
                           )
        
        
        # CheckListEditor display with four columns
        random_group = Group(Group(
                              Item('randomLibrary', 
                                   style='custom',  show_label=False),
                            seedsGroup,
                            ),            
                            label='Random Numbers'
                            ) 

        energy_group = Group( Item('external_grid_energy'),
                              Item('e0max_0', label= "Maximum allowable energy to start a run"),
                              Item('e0max_1', label="Maximum number of retries"),
                              Item('calculate_internal_electrostatic_energy'),
                              label='Energy Parameters', show_border = True,
                            ) 

        step_group = Group( Item('tstep', label="Translation (Angstrom/step)"),
                            Item('qstep', label="Quaternion (Degree/step)"),
                            Item('dstep', label="Torsion (Degree/step)"),
                              label='Step Size', show_border = True,
                            ) 

        miscOutputGroup = Group(Item('analysis_flag', label = "Perform cluster analysis:"),
                                 Item('write_all_flag', label = "Write all conformations in a cluster:"),
                                 Item('rmstol', label = "RMS Cluster Tolerance (Angstrom):"),
                                 Item(name='rmsatoms', style='custom', label="For RMS calculation use:"),
                                 Item('rmsref',  label = 'Reference structure file for RMS'),
                                 )
        output_group = VGroup( Item('outputLevelEnum', style='custom', show_label=False),
                               miscOutputGroup, 
                              label='Output Options'
                            ) 
        
        custom_parameter_group = Group(Item(name='custom_parameter_file', label = "Use custom parameter library"), 
           Item(name='parameter_file', enabled_when='object.custom_parameter_file'))
        
        include_1_4_interactions_group = Group(
           Item(name='include_1_4_interactions_flag', label='Include internal 1-4 interactions'),
           Item(name='include_1_4_interactions', label='Scaling factor for 1-4 interactions',
                enabled_when='object.include_1_4_interactions_flag'),
                        ) 
        
        
#        unbound_group = Group(Item(name='unbound_model_flag', label = "Include unbound ligand energy"), 
#           Item(name='unbound_model', enabled_when='object.unbound_model_flag == True', label="Unbound ligand state"),
#           label='Unbound Ligand Parameters', show_border = True,
#           )
#        
        
        epdb_group = Group(Item(name='epdb_flag', label = "Compute binding energy without docking (epdb)"), 
               Item(name='epdb', enabled_when='object.epdb_flag', label = "Enter filename for epdb calculation"))
        
        
        autodock4_specific_group = Group(custom_parameter_group, include_1_4_interactions_group,
                                         epdb_group,
                                         label='New in AutoDock 4.2'
                                         )
        
        self.docking_parameters_group =  Group(  output_group,
                       Group(
                       energy_group, 
                       step_group, label="Energy and Steps"),
                       random_group, 
                      autodock4_specific_group, 
                      layout='tabbed',
                      label = 'Docking Parameters',
                      ) 
        
        self.docking_parameters_view = View(  output_group,
                       Group(
                       energy_group, 
                       step_group, label="Energy and Steps"),
                       random_group, 
                      autodock4_specific_group, 
                      title = 'AutoDock Parameters', buttons = Buttons
                      ) 
        
    def _randomLibrary_changed(self, old, new):
        if new == self.randomLibraryTuple[0]:
            if self.seedType1 == 'custom':
                seed1 = self.seedInt1
            else:
                seed1 = self.seedType1.lower()
            if self.seedType2 == 'custom':
                seed2 = self.seedInt2
            else:
                seed2 = self.seedType2.lower()    
            self.dpo['seed']['value'] = [seed1, seed2]
        else:
            if self.seedType1 == 'custom':
                seed1 = self.seedInt1
            else:
                seed1 = self.seedType2.lower()
            self.dpo['seed']['value'] = [seed1]

    def _seedType1_changed(self):  
        self._randomLibrary_changed(None, self.randomLibrary)
        
    def _seedType2_changed(self):  
        self._randomLibrary_changed(None, self.randomLibrary)
    
    def _external_grid_energy_changed(self, new):
        self.dpo['extnrg']['value'] = new

    def _e0max_0_changed(self, new):
        self.dpo['e0max']['value'][0] = new
        
    def _e0max_1_changed(self, new):
        self.dpo['e0max']['value'][1] = new
        
    def _calculate_internal_electrostatic_energy_changed(self, new):
        self.dpo['intelec']['value'] = new
        
    def _dstep_changed(self, new):
        self.dpo['dstep']['value'] = new
        
    def _tstep_changed(self, new):
        self.dpo['tstep']['value'] = [new]

    def _qstep_changed(self, new):
        self.dpo['qstep']['value'] = new

    def _outputLevelEnum_changed(self, new):
        index = self.outputLevelList.index(new)
        self.dpo['outlev']['value'] = index
        
    def _write_all_flag_changed(self, new):
        self.dpo['write_all_flag']['value'] = new
    
    def _analysis_changed(self, new):
        self.dpo['analysis']['value'] = new
        
    def _rmstol_changed(self, new):
        self.dpo['rmstol']['value'] = new

    def _rmsatoms_changed(self, new):
        if new:
            self.dpo['rmsatoms_flag']['value'] = True
            self.dpo['rmsatoms']['value'] = new
        else:
            self.dpo['rmsatoms_flag']['value'] = False            
            self.dpo['rmsatoms']['value'] = new
            

    def _custom_parameter_file_changed(self, new):
        self.dpo['custom_parameter_file']['value'] = new
    
    def _parameter_file_changed(self, new):
        self.dpo['parameter_file']['value'] = new

    def _include_1_4_interactions_flag_changed(self, new):
        self.dpo['include_1_4_interactions_flag']['value'] = new
    
    def _include_1_4_interactions_changed(self, new):
        self.dpo['include_1_4_interactions']['value'] = new
    
    def _unbound_model_flag_changed(self, new):
        self.dpo['unbound_model_flag']['value'] = new

    def _unbound_model_changed(self, new):
        self.dpo['unbound_model']['value'] = new
        
    def _epdb_flag_changed(self, new):
        self.dpo['epdb_flag']['value'] = new
        
    def _epdb_changed(self, new):
        self.dpo['epdb']['value'] = new      


class TraitedGeneticAlgorithmParameters(TraitedDockingParameters): 
    ga_run = PositiveInt()
    ga_pop_size = PositiveInt()
    ga_num_evals = PositiveInt()
    ga_evals_type = Trait('short', 'medium', 'long', 'custom')
    ga_evals_type.default_value(0,"short")    

    ga_num_generations = PositiveInt()
    ga_elitism = PositiveInt()
    ga_mutation_rate = Float
    ga_crossover_rate = Float
    ga_crossover_mode = Enum('twopt', 'arithmetic', 'uniform')
    ga_cauchy_alpha = Float
    ga_cauchy_beta = Float    
    ga_window_size = Float
    def __init__(self):
        super(TraitedGeneticAlgorithmParameters, self).__init__()
        self.ga_run = self.dpo['ga_run']['value']
        self.ga_pop_size = self.dpo['ga_pop_size']['value']
        self.ga_num_evals = 250000 #self.dpo['ga_num_evals']['value']
        self.ga_num_generations = self.dpo['ga_num_generations']['value']
        self.ga_elitism = self.dpo['ga_elitism']['value']
        self.ga_mutation_rate = self.dpo['ga_mutation_rate']['value']
        self.ga_crossover_rate = self.dpo['ga_crossover_rate']['value']
        self.ga_crossover_mode = self.dpo['ga_crossover_mode']['value']
        self.ga_cauchy_alpha = self.dpo['ga_cauchy_alpha']['value']
        self.ga_cauchy_beta = self.dpo['ga_cauchy_beta']['value']
        self.ga_window_size = self.dpo['ga_window_size']['value']
        

        ga_evals_group = HGroup(spring, Item(name='ga_evals_type', style='simple', label="Maximum number of energy evaluations"), 
                                Item(name='ga_num_evals', show_label=False, enabled_when='object.ga_evals_type == "custom"'), 
                                )    
        self.genetic_algorithm_group = Group(Item(name='ga_run', label="Number of GA runs"),
                                             Item(name='ga_pop_size', label="Number of individuals in population"),
                                             #Item(name='ga_num_evals', label="Maximum number of energy evaluations"),
                                             ga_evals_group,                
                                             Item(name='ga_num_generations', label="Maximum number of generations"),
                                             Item(name='ga_elitism', label="Number of top individuals to survive to next generation"),
                                             Item(name='ga_mutation_rate', label="Rate of gene mutation"),
                                             Item(name='ga_crossover_rate', label="Rate of crossover"),
                                             Item(name='ga_crossover_mode', label="GA crossover mode"),
                                             Item(name="ga_cauchy_alpha", label="Mean of Cauchy distribution for gene mutation (Cauchy alpha)"),
                                             Item(name="ga_cauchy_beta", label="Variance of Cauchy distribution for gene mutation (Cauchy beta)"),
                                             Item(name='ga_window_size', label="Number of generations for picking worst individual (GA window size)"),
                                             label = 'Genetic Algorithm Parameters',                                                   
                                             )
        self.genetic_algorithm_view = View(self.genetic_algorithm_group, buttons = Buttons) 

    def _ga_run_changed(self, new):
        self.dpo['ga_run']['value'] = new      
        
    def _ga_pop_size_changed(self, new):
        self.dpo['ga_pop_size']['value'] = new      
    
    def _ga_num_evals_changed(self, new):
        self.dpo['ga_num_evals']['value'] = new
              
    def _ga_evals_type_changed(self, new):
        if new == 'short':
            self.ga_num_evals = 250000
        elif new == 'medium':
            self.ga_num_evals = 2500000
        elif new == 'long':
            self.ga_num_evals = 25000000
            
    def _ga_num_generations_changed(self, new):
        self.dpo['ga_num_generations']['value'] = new      
        
    def _ga_elitism_changed(self, new):
        self.dpo['ga_elitism']['value'] = new      

    def _ga_mutation_rate_changed(self, new):
        self.dpo['ga_mutation_rate']['value'] = new      

    def _ga_crossover_rate_changed(self, new):
        self.dpo['ga_crossover_rate']['value'] = new      

    def _ga_cauchy_alpha_changed(self, new):
        self.dpo['ga_cauchy_alpha']['value'] = new      

    def _ga_cauchy_beta_changed(self, new):
        self.dpo['ga_cauchy_beta']['value'] = new      

    def _ga_window_size_changed(self, new):
        self.dpo['ga_window_size']['value'] = new      


class TraitedSimulatedAnnealingParameters(TraitedDockingParameters): 
    runs = PositiveInt()
    cycles = PositiveInt()
    accs = PositiveInt()
    rejs = PositiveInt()
    select = Enum("Minimum state","Last state")
    linear_schedule = Enum("Linear","Geometric")
    trnrf = Float
    quarf = Float
    dihrf = Float
    rtrf = Float
    rt0 = Float
    def __init__(self):
        super(TraitedSimulatedAnnealingParameters, self).__init__()
        self.runs = self.dpo['runs']['value']
        self.cycles = self.dpo['cycles']['value']
        self.accs = self.dpo['accs']['value']
        self.rejs = self.dpo['rejs']['value']
        self.trnrf = self.dpo['trnrf']['value']
        self.quarf = self.dpo['quarf']['value']
        self.dihrf = self.dpo['dihrf']['value']
        self.rtrf = self.dpo['rtrf']['value']
        self.rt0 = self.dpo['rt0']['value']
        
        #self.ga_pop_size = 
        self.simulated_annealing_group = Group(Item(name='runs', label="Number of runs"),
                                             Item(name='cycles', label="Number of temperature reduction cycles"),
                                             Item(name='accs', label="Maximum number of accepted steps per cycle"),
                                             Item(name='rejs', label="Maximum number of rejected steps per cycle"),
                                             Item(name='select', label="State selection for the next cycle", style='custom'),
                                             Item(name='linear_schedule', label="Temperature reduction schedule type", style='custom'),
                                             Item(name='trnrf', label="Per cycle reduction factor for translation"),
                                             Item(name='quarf', label="Per cycle reduction factor for quaternion"),
                                             Item(name='dihrf', label="Per cycle reduction factor for dihedral"),
                                             Item(name='rtrf', label="Per cycle reduction factor for temperature"),
                                             Item(name='rt0', label="Initial annealing temperature"),
                                             label = 'Simulated Annealing Parameters',                                                   
                                             )

        self.simulated_annealing_view = View(self.simulated_annealing_group, buttons = Buttons) 
    
    def _runs_changed(self, new):
        self.dpo['runs']['value'] = new      
    
    def _cycles_changed(self, new):
        self.dpo['cycles']['value'] = new      

    def _accs_changed(self, new):
        self.dpo['accs']['value'] = new      
        
    def _rejs_changed(self, new):
        self.dpo['rejs']['value'] = new      
        
    def _select_changed(self, new):
        if new == "Minimum state":
            self.dpo['select']['value'] = 'm'
        else:
            self.dpo['select']['value'] = 'l' #(l)ast state

    def _linear_schedule_changed(self, new):
        if new == "Linear":
            self.dpo['linear_schedule']['value'] = 1
        else:
            self.dpo['linear_schedule']['value'] = 0
            
    def _trnrf_changed(self, new):
        self.dpo['trnrf']['value'] = new      

    def _quarf_changed(self, new):
        self.dpo['quarf']['value'] = new      

    def _dihrf_changed(self, new):
        self.dpo['dihrf']['value'] = new      

    def _rtrf_changed(self, new):
        self.dpo['rtrf']['value'] = new      

    def _rt0_changed(self, new):
        self.dpo['rt0']['value'] = new      


class TraitedLocalSearchParameters(TraitedDockingParameters): 
    do_local_only = PositiveInt()
    sw_max_its = PositiveInt()
    sw_max_succ = PositiveInt()
    sw_max_fail = PositiveInt()
    sw_rho = Float
    sw_lb_rho = Float
    ls_search_freq = Float
    sw_type = Enum("Uniform","Relative")
    
    def __init__(self):
        super(TraitedLocalSearchParameters, self).__init__()
        self.do_local_only = self.dpo['do_local_only']['value']
        self.sw_max_its = self.dpo['sw_max_its']['value']
        self.sw_max_succ = self.dpo['sw_max_succ']['value']
        self.sw_max_fail = self.dpo['sw_max_fail']['value']
        self.sw_rho = self.dpo['sw_rho']['value']
        self.sw_lb_rho = self.dpo['sw_lb_rho']['value']
        self.ls_search_freq = self.dpo['ls_search_freq']['value']
  
        #self.ga_pop_size = 
        self.local_search_group = Group(Item(name='do_local_only', label="Number of LS runs"),
                                             Item(name='sw_max_its', label="Maximum number of iterations"),
                                             Item(name='sw_max_succ', label="Maximum number of successes in a row before changing rho"),
                                             Item(name='sw_max_fail', label="Maximum number of failures in a row before changing rho"),
                                             Item(name='sw_rho', label="Solis and Wets parameter defining initial variance \nand size of local space to sample (rho)"),
                                             Item(name='sw_lb_rho', label="Lower bound on rho"),
                                             Item(name='ls_search_freq', label="Probability of any particular phenotype being\nsubjected to local search"),
                                             Item(name='sw_type', label="Solis and Wets variances", style='custom'),
                                             label = 'Local Search Parameters',                                                   
                                             )

        self.local_search_view = View(self.local_search_group, buttons = Buttons) 
    
    def _do_local_only_changed(self, new):
        self.dpo['do_local_only']['value'] = new      
    
    def _sw_max_its_changed(self, new):
        self.dpo['sw_max_its']['value'] = new      

    def _sw_max_succ_changed(self, new):
        self.dpo['sw_max_succ']['value'] = new      
        
    def _sw_max_fail_changed(self, new):
        self.dpo['sw_max_fail']['value'] = new      

    def _sw_rho_changed(self, new):
        self.dpo['sw_rho']['value'] = new      
            
    def _sw_lb_rho_changed(self, new):
        self.dpo['sw_lb_rho']['value'] = new      

    def _dihrf_changed(self, new):
        self.dpo['dihrf']['value'] = new      

    def _rtrf_changed(self, new):
        self.dpo['rtrf']['value'] = new      

    def _rt0_changed(self, new):
        self.dpo['rt0']['value'] = new      
        
    def _sw_type_changed(self, new):
        if new == "Uniform":
            self.dpo['set_sw1']['value'] = 1
            self.dpo['set_psw1']['value'] = 0
        else:
            self.dpo['set_sw1']['value'] = 0
            self.dpo['set_psw1']['value'] = 1

class TraitedLamarckianGAParameters(TraitedGeneticAlgorithmParameters, TraitedLocalSearchParameters):
    pass

import  wx
class BookDialog(wx.Dialog):
    def __init__(self, parent, parameters, title):
        wx.Dialog.__init__(self, parent, -1, title)
        sizer = wx.BoxSizer(wx.VERTICAL)
        self.book = wx.Treebook(self, -1, style=
                            wx.BK_DEFAULT
                            #wx.BK_TOP
                            #wx.BK_BOTTOM
                            #wx.BK_LEFT
                            #wx.BK_RIGHT
                            )        
        sizer.Add(self.book)
        self.pages = []
        self.parameters = parameters
        self.sizer = sizer
        
    def finish_layout(self):
        line = wx.StaticLine(self, -1, size=(20,-1), style=wx.LI_HORIZONTAL)
        self.sizer.Add(line, 0, wx.GROW|wx.ALIGN_CENTER_VERTICAL, 5) 
        
        self.okButton = wx.Button(self, wx.ID_OK, "")
        self.cancelButton = wx.Button(self, wx.ID_CANCEL, "") 
        buttonSizer = wx.BoxSizer(wx.HORIZONTAL)
        lin = wx.StaticLine(self)
        buttonSizer.Add((10, -1), 1, flag=wx.EXPAND | wx.ALIGN_RIGHT)
        buttonSizer.Add(self.okButton, 0, wx.ALIGN_RIGHT|wx.ALL, 5)
        buttonSizer.Add(self.cancelButton, 0, wx.ALIGN_RIGHT|wx.ALL, 5)
        self.sizer.Add(lin,0,wx.EXPAND)
        self.sizer.Add(buttonSizer, 0, wx.EXPAND|wx.ALIGN_BOTTOM)
        
        self.SetSizer(self.sizer)
        self.sizer.Fit(self)        
        self.Bind(wx.EVT_BUTTON, self._on_ok, self.okButton)
        self.Bind(wx.EVT_BUTTON, self._on_cancel, self.cancelButton)


    def _on_error(self, errors):
        """ Handles editing errors.
        """
        self.okButton.Enable(errors == 0)
        
    def _on_ok(self, event=None):
        """ Handles the user clicking the **OK** button.
        """
        for page in self.pages:
            if page.handler.close(page.info, True):            
                page.finish()
        self.EndModal(wx.ID_OK)
        

    def _on_cancel (self, event):
        """ Handles a request to cancel all changes.
        """
        for page in self.pages:
            page.handler.close(page.info, False)
            page.finish()  
        self.EndModal(wx.ID_CANCEL)

class GeneticAlgorithmParametersGUI(BookDialog):
    def __init__(self, parent, parameters):
        BookDialog.__init__(self, parent, parameters, 'Genetic Algorithm Parameters')

        page1 = parameters.genetic_algorithm_view.ui(parameters, self.book, kind='subpanel')
        page1.on_trait_change( self._on_error, 'errors', dispatch = 'ui' ) 
        self.pages.append(page1)        
        self.book.AddPage(page1.control, "Genetic Algorithm")

        page2 = parameters.docking_parameters_view.ui(parameters, self.book, kind='subpanel')
        page2.on_trait_change( self._on_error, 'errors', dispatch = 'ui' )        
        self.pages.append(page2)       
        self.book.AddPage(page2.control, "Docking Parameters")
        self.finish_layout()
        
class SimulatedAnnealingParametersGUI(BookDialog):
    def __init__(self, parent, parameters):
        BookDialog.__init__(self, parent, parameters, 'Simulated Annealing Parameters')

        page1 = parameters.simulated_annealing_view.ui(parameters, self.book, kind='subpanel')
        page1.on_trait_change( self._on_error, 'errors', dispatch = 'ui' ) 
        self.pages.append(page1)        
        self.book.AddPage(page1.control, "Simulated Annealing")
        
        page2 = parameters.docking_parameters_view.ui(parameters, self.book, kind='subpanel')
        page2.on_trait_change( self._on_error, 'errors', dispatch = 'ui' )        
        self.pages.append(page2)       
        self.book.AddPage(page2.control, "Docking Parameters")
        self.finish_layout()
 
class LocalSearchParametersGUI(BookDialog):
    def __init__(self, parent, parameters):
        BookDialog.__init__(self, parent, parameters, 'Local Search Parameters')
        page1 = parameters.local_search_view.ui(parameters, self.book, kind='subpanel')
        page1.on_trait_change( self._on_error, 'errors', dispatch = 'ui' ) 
        self.pages.append(page1)
        
        self.book.AddPage(page1.control, "Local Search")
        page2 = parameters.docking_parameters_view.ui(parameters, self.book, kind='subpanel')
        page2.on_trait_change( self._on_error, 'errors', dispatch = 'ui' )        
        self.pages.append(page2)       
        self.book.AddPage(page2.control, "Docking Parameters")
        self.finish_layout()

class LamarckianGAParametersGUI(BookDialog):
    def __init__(self, parent, parameters):
        BookDialog.__init__(self, parent, parameters, 'Lamarckian Genetic Algorithm Parameters')
        page1 = parameters.genetic_algorithm_view.ui(parameters, self.book, kind='subpanel')
        page1.on_trait_change( self._on_error, 'errors', dispatch = 'ui' ) 
        self.pages.append(page1)
        self.book.AddPage(page1.control, "Genetic Algorithm")
        
        page2 = parameters.local_search_view.ui(parameters, self.book, kind='subpanel')
        page2.on_trait_change( self._on_error, 'errors', dispatch = 'ui' )        
        self.pages.append(page2)
        self.book.AddPage(page2.control, "Local Search")
        
        page3 = parameters.docking_parameters_view.ui(parameters, self.book, kind='subpanel')
        page3.on_trait_change( self._on_error, 'errors', dispatch = 'ui' )        
        self.pages.append(page3)           
        self.book.AddPage(page3.control, "Docking Parameters")
        self.finish_layout()
           
#TODO: pass file to configure_traits to save the state

if __name__ == "__main__":
    demo = TraitedDockingParameters()
    demo.configure_traits(view=demo.docking_parameters_view)