#$Id: mayaviActions.py 16 2009-11-16 21:44:21Z sarkiss $

#Customized from tvtk/plugins/scene/ui/actions.py by Sargis Dallakyan (sargis@scripps.edu)

# Enthought library imports.
from enthought.pyface.api import FileDialog, OK
from enthought.pyface.action.api import Action
from enthought.traits.api import Instance, Property, Str

class MyAction(Action):
    def __init__(self, frame):
        self.frame = frame
    
class SceneAction(MyAction):
    """ Base class for actions that require a scene manager. """

    #### 'SceneAction' interface ##############################################

    # The scene manager.
    scene_manager = Property(Instance(
        'enthought.tvtk.plugins.scene.i_scene_manager import ISceneManager'
    ))

    def _get_scene_manager(self):
        """ Trait property getter. """

        from enthought.tvtk.plugins.scene.i_scene_manager import (
            ISceneManager
        )
        
        return self.window.get_service(ISceneManager)



class SaveScene(SceneAction):
    """ An action that saves a scene to an image. """

    #### 'Action' interface ###################################################

    name = 'Save Scene'

    def perform(self, event):
        """ Performs the action. """        

        extensions = [
            '*.png', '*.jpg', '*.jpeg', '*.tiff', '*.bmp', '*.ps', '*.eps',
            '*.tex', '*.rib', '*.wrl', '*.oogl', '*.pdf', '*.vrml', '*.obj',
            '*.iv'
        ]

        wildcard = '|'.join(extensions)

        dialog = FileDialog(
            parent   = self.frame,
            title    = 'Save scene to image',
            action   = 'save as',
            wildcard = wildcard
        )
        if dialog.open() == OK:
            scene = self.frame.mayaviEngine.scene
            if scene is not None:
                scene.save(dialog.path)

        return
    

class SaveSceneToImage(SceneAction):
    """ An action that saves a scene to an image. """

    #### 'Action' interface ###################################################
    
    # Name of the action.
    name = 'Image'

    #### 'SaveSceneToImage' interface #########################################
    
    # The save method name.
    save_method = Str('save')

    # The wildcard for the file dialog.
    wildcard = Str("All files (*.*)|*.*")

    ###########################################################################
    # 'Action' interface.
    ###########################################################################

    def perform(self, event):
        """ Perform the action. """        

        dialog = FileDialog(
            parent   = self.frame,
            title    = 'Save scene to %s' % self.name,
            action   = 'save as',
            wildcard = self.wildcard
        )
        if dialog.open() == OK:
            scene = self.frame.mayaviEngine.scene
            if scene is not None:
                method = getattr(scene, self.save_method)
                method(dialog.path)

        return


# These are all specific subclasses that save particular images.
class SaveSceneToPNG(SaveSceneToImage):
    name        = 'PNG Image'
    save_method = 'save_png'
    wildcard    = 'PNG images (*.png)|*.png|' \
                  'All files (*.*)|*.*'
    
class SaveSceneToJPEG(SaveSceneToImage):
    name        = 'JPEG Image'
    save_method = 'save_jpg'
    wildcard    = 'JPEG images (*.jpg)|*.jpg|' \
                  'JPEG images (*.jpeg)|*.jpeg|' \
                  'All files (*.*)|*.*'

class SaveSceneToBMP(SaveSceneToImage):
    name        = 'BMP Image'
    save_method = 'save_bmp'
    wildcard    = 'BMP images (*.bmp)|*.bmp|' \
                  'All files (*.*)|*.*'

class SaveSceneToTIFF(SaveSceneToImage):
    name        = 'TIFF Image'
    save_method = 'save_tiff'
    wildcard    = 'TIFF images (*.tif)|*.tif|' \
                  'TIFF images (*.tiff)|*.tiff|' \
                  'All files (*.*)|*.*'

class SaveSceneToPS(SaveSceneToImage):
    name        = 'PostScript bitmap Image'
    save_method = 'save_ps'
    wildcard    = 'PostScript bitmap images (*.ps)|*.ps|' \
                  'All files (*.*)|*.*'

class SaveSceneToGL2PS(SaveSceneToImage):
    name        = 'Vector PS/EPS/PDF/TeX'
    save_method = 'save_gl2ps'
    wildcard    = 'All files (*.*)|*.*|' \
                  'EPS files (*.eps)|*.eps|' \
                  'PS files (*.ps)|*.ps|' \
                  'PDF files (*.pdf)|*.pdf|' \
                  'TeX files (*.tex)|*.tex'

class SaveSceneToRIB(SaveSceneToImage):
    name        = 'RenderMan RIB file'
    save_method = 'save_rib'
    wildcard    = 'RIB files (*.rib)|*.rib|' \
                  'All files (*.*)|*.*'

class SaveSceneToOOGL(SaveSceneToImage):
    name        = 'GeomView OOGL file'
    save_method = 'save_oogl'
    wildcard    = 'OOGL files (*.oogl)|*.oogl|' \
                  'All files (*.*)|*.*'

class SaveSceneToIV(SaveSceneToImage):
    name        = 'OpenInventor file'
    save_method = 'save_iv'
    wildcard    = 'OpenInventor files (*.iv)|*.iv|' \
                  'All files (*.*)|*.*'

class SaveSceneToVRML(SaveSceneToImage):
    name        = 'VRML file'
    save_method = 'save_vrml'
    wildcard    = 'VRML files (*.wrl)|*.wrl|' \
                  'All files (*.*)|*.*'

class SaveSceneToOBJ(SaveSceneToImage):
    name        = 'Wavefront OBJ file'
    save_method = 'save_wavefront'
    wildcard    = 'OBJ files (*.obj)|*.obj|' \
                  'All files (*.*)|*.*'

import sys 
from os.path import isfile



# Local imports
from enthought.mayavi.core.common import error, exception


######################################################################
# `SaveVisualization` class.
######################################################################
class SaveVisualization(MyAction):
    """ An action that saves the current visualization. """

    tooltip       = "Save current visualization"

    description   = "Save current visualization to a MayaVi2 file"
    name = "Save Visualization"
    ###########################################################################
    # 'Action' interface.
    ###########################################################################

    def perform(self, event):
        """ Performs the action. """
        wildcard = 'MayaVi2 files (*.mv2)|*.mv2|' + FileDialog.WILDCARD_ALL
        dialog = FileDialog(parent=self.frame,
                            title='Save MayaVi2 file',
                            action='save as', wildcard=wildcard
                            )
        if dialog.open() == OK:
            self.frame.mayaviEngine.engine.save_visualization(dialog.path)

######################################################################
# `LoadVisualization` class.
######################################################################
class LoadVisualization(MyAction):
    """ An action that loads a visualization from file. """

    tooltip       = "Load saved visualization"

    description   = "Load saved visualization from a MayaVi2 file"
    name = "Load Visualization"
    ###########################################################################
    # 'Action' interface.
    ###########################################################################

    def perform(self, event):
        """ Performs the action. """
        wildcard = 'MayaVi2 files (*.mv2)|*.mv2|' + FileDialog.WILDCARD_ALL
        parent = self.frame
        dialog = FileDialog(parent=parent,
                            title='Open MayaVi2 file',
                            action='open', wildcard=wildcard
                            )
        if dialog.open() == OK:
            if not isfile(dialog.path):
                error("File '%s' does not exist"%dialog.path, parent)
                return
            self.frame.mayaviEngine.engine.load_visualization(dialog.path)

######################################################################
# `RunScript` class.
######################################################################
class RunScript(MyAction):
    """ An action that runs a mayavi script.  
    
    WARNING: this can be dangerous since the file runs execfile! """

    tooltip       = "Execute a Python script (typically a Mayavi script)"

    description   = "Execute a Python script (typically a Mayavi script)"

    ###########################################################################
    # 'Action' interface.
    ###########################################################################

    def perform(self, event):
        """ Performs the action. """
        wildcard = 'Python files (*.py)|*.py'
        parent = self.frame
        dialog = FileDialog(parent=parent,
                            title='Open Python file',
                            action='open', wildcard=wildcard
                            )
        if dialog.open() == OK:
            if not isfile(dialog.path):
                parent.log.error("File '%s' does not exist"%dialog.path)
                return
            
            # Get the globals.
            # The following code is taken from scripts/mayavi2.py.
            g = sys.modules['__main__'].__dict__
            execfile(dialog.path, g, g)

from enthought.pyface.action.api import MenuManager
import wx
ID_RUN = wx.NewId()
def AddMayaviMenus(menu, frame):
    
    runScript = RunScript(frame)
    menu.Append(ID_RUN, "&Run Python Script")
    frame.Bind(wx.EVT_MENU, runScript.perform, id=ID_RUN)
    #mayaviMenu = MenuManager(SaveVisualization(frame), LoadVisualization(frame))
    #menu.AppendMenu(-1, "Mayavi", mayaviMenu.create_menu(frame))
    saveMenu = MenuManager(SaveSceneToPNG(frame), SaveSceneToJPEG(frame),SaveSceneToBMP(frame), SaveSceneToTIFF(frame),
                           SaveSceneToPS(frame), SaveSceneToGL2PS(frame), SaveSceneToRIB(frame), SaveSceneToOOGL(frame),
                           SaveSceneToIV(frame), SaveSceneToVRML(frame),  SaveSceneToOBJ(frame))
    
    menu.AppendMenu(-1, "S&ave Scene As", saveMenu.create_menu(frame))

